<?php

namespace App\Http\Controllers\api\partneruser;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use App\Models\Fare;
use App\Models\Zone;
use App\Models\SearchHistory;
use App\Models\VehicleCategory;
use App\Models\BrandVehicle;
use App\Models\Vehicle;
use App\Models\Vendors;
use App\Models\Settings;
use App\Models\Review;
use App\Models\BookingHistory;
use App\Models\TripLifeCycle;
use App\Models\Banners;
use App\Models\Notifications;
use Illuminate\Database\QueryException;
use Throwable;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;
use Razorpay\Api\Api;

class FareController extends Controller
{
    public function calculateFare(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'origin' => 'required',
                'destination' => 'required',
                'vehicle_type' => 'required',
                'waypoints' => 'required',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $origin = $request->origin;
            $destination = $request->destination;
            $vehicleType = $request->vehicle_type;
            $waypoints = $request->waypoints ?? [];

            // Step 1: Get coordinates of origin
            $originCoords = $this->getCoordinatesFromAddress($origin);

            // Step 2: Find the zone for origin
            $zones = Zone::all();
            $originZone = null;
            foreach ($zones as $zone) {
                $polygon = json_decode($zone->coordinates, true); // ← fix here
                if ($this->isPointInPolygon($originCoords, $polygon)) {
                    $originZone = $zone;
                    break;
                }
            }


            if (!$originZone) {
                return response()->json(['error' => 'Origin is outside service zones'], 422);
            }
            $setting = Settings::first();
            $additional_km_in_table = $setting->additional_km;

            // Total days 
            $departure = Carbon::parse(date("Y-m-d", strtotime($request->departure_date)));
            $return = Carbon::parse(date("Y-m-d", strtotime($request->return_date)));
            $totalDays = $departure->diffInDays($return) + 1; // +1 to include both start and end dates
            $distance = $this->getTotalDistanceFromGoogle($origin, $destination, $waypoints) + ($totalDays * $additional_km_in_table);
          
            // if ($request->filled('waypoints')) {
            //     $json = $request->waypoints;
            //     $locations = json_decode($json, true); // convert to PHP array

            //     $count = count($locations);

            //     // Step 3: Get total distance
            //     $distance = $this->getTotalDistanceFromGoogle($origin, $destination, $waypoints) + ($count * $additional_km_in_table);
            // } else {
            //     $waypoints = [];
            //     $distance = $this->getTotalDistanceFromGoogle($origin, $destination, $waypoints) + $additional_km_in_table;
            // }
            // Step 4: Calculate fare
            $fare = Fare::where('vehicle_type_id', $vehicleType)
                ->whereRaw('json_contains(city_id, \'["' . $originZone->id . '"]\')')
                ->first();

            // $fare = Fare::where('vehicle_type_id', $vehicleType)
            //     ->whereJsonContains('city_id', $originZone->id)
            //    ->first();
            // ->toSql();

            //     print_r($fare);die();


            if (!$fare) {
                $fare->fare = 0;
                $fare->tax_amount = 0;
                $fare->total_rate = 0;
                $fare->platform_fee = 0;
                // return response()->json(['error' => 'No fare defined for this vehicle type and zone'], 404);
            }

            $offerStartDate = $fare->offer_start_date;
            $offerEndDate = $fare->offer_end_date;
            $bookingDate = date("Y-m-d", strtotime($request->departure_date));
            $discountPercentage = 20;

            $surchargeStartDate = $fare->surcharge_start_date;
            $surchargeEndDate = $fare->surcharge_end_date;
            $surchargePercentage = 10;

            $baseKm = $fare->base_km * $totalDays;
            $baseRate = $fare->base_rate;
            if ($bookingDate >= $offerStartDate && $bookingDate <= $offerEndDate) {
                // Apply discount
                $discountAmount = $baseRate * ($discountPercentage / 100);
                $offerPrice = $baseRate - $discountAmount;
                $baseRate = round($offerPrice, 2);
            }


            if ($bookingDate >= $surchargeStartDate && $bookingDate <= $surchargeEndDate) {
                // Apply discount
                $discountAmount = $baseRate * ($surchargePercentage / 100);
                $surchargePrice = $baseRate + $discountAmount;
                $baseRate = round($surchargePrice, 2);
            }
            // if(auth()->user()->role=='Partner'){
            //     $baseRate = $fare->base_rate*(auth()->user()->platform_fee/100);
            // }else{

            //  }

            $additionalKmRate = $fare->additional_km_rate;

            $totalFare = $baseRate * $totalDays;

            if ($distance > $baseKm) {
                $extraKm = $distance - $baseKm;
                $totalFare += $extraKm * $additionalKmRate;
            }
            $taxinno = ($setting->tax / 100);
            $tax = ($setting->tax / 100) * $totalFare;

            $commission_rate =  ($request->commission / 100) * $totalFare;
            $bfare = round($totalFare + $commission_rate, 2);
            $tax_amount = round($tax, 2);
            $total_rate = $bfare + $tax_amount;
            $platform_fee = ($setting->platform_fee / 100) * $total_rate;


            // $total_fare =$total_rate + $commission_rate ;
            // $platform_fee= ($setting->platform_fee/100) * $total_fare;
            return response()->json([
                'distance_km' => round($distance, 2),
                'base_fare' => round($bfare, 2),
                'tax' => round($tax_amount, 2),
                'commission_rate' => round($commission_rate, 2),
                'subtotal' => round($total_rate, 1),
                'paltform_fee' => round($platform_fee, 1),

            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getTotalDistanceFromGoogle($origin, $destination, $waypoints = [])
    {
        try {
            $apiKey = Settings::pluck('google_map_api_key')->first();
            if (!empty($waypoints)) {
                $waypoints = is_array($waypoints) ? $waypoints : explode('","', $waypoints ?? '');
                $lastWaypoint = end($waypoints);
                $waypointsStr = implode('|', $waypoints);
                $response = Http::get("https://maps.googleapis.com/maps/api/directions/json", [
                    'origin' => $origin,
                    'destination' => $destination,
                    'waypoints' => $waypointsStr,
                    'key' => $apiKey,
                ]);
            } else {
                $response = Http::get("https://maps.googleapis.com/maps/api/directions/json", [
                    'origin' => $origin,
                    'destination' => $destination,
                    'key' => $apiKey
                ]);
            }
            $data = $response->json();

            if (isset($data['routes'][0]['legs'])) {
                $totalDistance = 0;

                foreach ($data['routes'][0]['legs'] as $leg) {
                    $totalDistance += $leg['distance']['value']; // in meters
                }

                $totalKm = $totalDistance / 1000;

                // find total return km
                $response = Http::get("https://maps.googleapis.com/maps/api/directions/json", [
                    'origin' => $lastWaypoint,
                    'destination' => $origin,
                    'key' => $apiKey
                ]);

                $data = $response->json();

                if ($data['status'] !== 'OK') {
                    return response()->json(['error' => 'Unable to calculate distance', 'status' => $data['status']], 500);
                }

                $distanceInMeters = $data['routes'][0]['legs'][0]['distance']['value'];
                $distanceInKm = round($distanceInMeters / 1000, 2);

                $totalDistance = $totalKm + $distanceInKm;
                // echo $totalDistance.'='.$totalKm .'+'.$distanceInKm;
                // die();
                return $totalDistance;
            }

            return response()->json([
                'error' => 'Unable to calculate distance'
            ], 400);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function isPointInPolygon($point, $polygon)
    {
        try {
            $lat = $point['lat'];
            $lng = $point['lng'];
            $inside = false;
            $j = count($polygon) - 1;

            for ($i = 0; $i < count($polygon); $i++) {
                $xi = $polygon[$i]['lat'];
                $yi = $polygon[$i]['lng'];
                $xj = $polygon[$j]['lat'];
                $yj = $polygon[$j]['lng'];

                $intersect = (($yi > $lng) != ($yj > $lng))
                    && ($lat < ($xj - $xi) * ($lng - $yi) / ($yj - $yi) + $xi);
                if ($intersect) $inside = !$inside;
                $j = $i;
            }

            return $inside;
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getCoordinatesFromAddress($address)
    {
        try {
            $apiKey = Settings::pluck('google_map_api_key')->first();
            $response = Http::get("https://maps.googleapis.com/maps/api/geocode/json", [
                'address' => $address,
                'key' => $apiKey,
            ]);

            $data = $response->json();
            if ($data['status'] === 'OK') {
                $location = $data['results'][0]['geometry']['location'];
                return ['lat' => $location['lat'], 'lng' => $location['lng']];
            }

            throw new \Exception("Invalid address or failed to fetch coordinates.");
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function chooseRide(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'origin' => 'required',
                'destination' => 'required',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }

            $setting = Settings::first();

            //     SearchHistory::insert([
            //         'user_id'=>auth()->user()->id,
            //         'departure_date'=>date("Y-m-d",strtotime($request->departure_date)),
            //         'return_date'=>date("Y-m-d",strtotime($request->return_date)),
            //         'departure_time'=>$request->departure_time,
            //         'return_time'=>$request->return_time,
            //         'pick_up'=>$request->origin,
            //         'drop_off'=>$request->destination,
            //         'stay_date'=>json_encode($request->stay_date),
            //         'stay_locations'=> json_encode($request->stay_locations),
            //         'adults'=>$request->adults,
            //         'children'=>$request->children,
            //         'client_name'=>$request->client_name,
            //         'client_mobile_number'=>$request->client_mobile_number,
            //         'client_email'=>$request->client_email,
            //     ]);

            $serachhistory = new SearchHistory();
            $serachhistory->user_id = auth()->user()->id;
            $serachhistory->departure_date = date("Y-m-d", strtotime($request->departure_date));
            $serachhistory->return_date = date("Y-m-d", strtotime($request->return_date));
            $serachhistory->departure_time = $request->departure_time;
            $serachhistory->return_time = $request->return_time;
            $serachhistory->pick_up = $request->origin;
            $serachhistory->drop_off = $request->destination;
            $serachhistory->stay_date = json_encode($request->stay_date);
            $serachhistory->stay_locations = json_encode($request->waypoints);
            $serachhistory->adults = $request->adults;
            $serachhistory->children = $request->children;
            $serachhistory->client_name = $request->client_name;
            $serachhistory->client_mobile_number = $request->client_mobile_number;
            $serachhistory->client_email = $request->client_email;
            $serachhistory->save();

            $origin = $request->origin;
            $destination = $request->destination;
            $waypoints = $request->waypoints ?? [];

            // Step 1: Get coordinates of origin
            $originCoords = $this->getCoordinatesFromAddress($origin);

            // Step 2: Find the zone for origin
            $zones = Zone::all();
            $originZone = null;
            foreach ($zones as $zone) {
                $polygon = json_decode($zone->coordinates, true); // ← fix here
                if ($this->isPointInPolygon($originCoords, $polygon)) {
                    $originZone = $zone;
                    break;
                }
            }


            if (!$originZone) {
                return response()->json(['error' => 'Origin is outside service zones'], 422);
            }

            $additional_km_in_table = $setting->additional_km;

            // Total days 
            $departure = Carbon::parse(date("Y-m-d", strtotime($request->departure_date)));
            $return = Carbon::parse(date("Y-m-d", strtotime($request->return_date)));
            $totalDays = $departure->diffInDays($return) + 1; // +1 to include both start and end dates
            $distance = $this->getTotalDistanceFromGoogle($origin, $destination, $waypoints) + ($totalDays * $additional_km_in_table);

            // if ($request->filled('waypoints')) {
            //     $json = $request->waypoints;
            //     $locations = json_decode($json, true); // convert to PHP array

            //     $count = count($locations);

            //     // Step 3: Get total distance
            //     $distance = $this->getTotalDistanceFromGoogle($origin, $destination, $waypoints) + ($totalDays * $additional_km_in_table);
            // } else {
            //     $waypoints = [];
            //     $distance = $this->getTotalDistanceFromGoogle($origin, $destination, $waypoints) + $additional_km_in_table;
            // }
            $vehicleType = VehicleCategory::where('status', 'Active')->get()->makeHidden(['status', 'created_at', 'updated_at']);

            foreach ($vehicleType  as $k => $item) {
                // Step 4: Calculate fare
                $item->gallery_images = [];
                $fare = Fare::where('vehicle_type_id', $item->id)
                    ->whereRaw('json_contains(city_id, \'["' . $originZone->id . '"]\')')
                    ->first();


                $item->image = asset("uploads/category/" . $item->image);
                $item->seats = BrandVehicle::where('vehicle_type', $item->id)->pluck('max_passengers')->first();
                $item->model_list = BrandVehicle::where('vehicle_type', $item->id)->pluck('model_name')->implode(', ');

                // 3 images
                $images = [];
                $brands = Vehicle::where('vehicle_type', $item->id)->get();

                $gallery_images = [];
                if (count($brands) > 0) {
                    foreach ($brands as $brand) {
                        if ($brand->image != NULL) {
                            foreach (json_decode($brand->image) as $i => $img) {
                                if ($i == 3) {
                                    break; // Stop looping once 3 images collected
                                }
                                $gallery_images[$i] = asset('uploads/vehicle/' . $img); // or Storage::url($img) if using storage



                            }
                        }
                    }
                }
                $item->gallery_images = $gallery_images;
                if (!$fare) {
                    $item->fare = round(0, 2);
                    $item->tax_amount = 0;
                    $item->total_rate = 0;
                    $item->platform_fee = 0;
                    continue;
                    // return response()->json(['error' => 'No fare defined for this vehicle type and zone'], 404);
                }

                $offerStartDate = $fare->offer_start_date;
                $offerEndDate = $fare->offer_end_date;
                $bookingDate = date("Y-m-d", strtotime($request->departure_date));
                $discountPercentage = 20;

                $surchargeStartDate = $fare->surcharge_start_date;
                $surchargeEndDate = $fare->surcharge_end_date;
                $surchargePercentage = 10;

                $baseKm = $fare->base_km * $totalDays;
                $baseRate = $fare->base_rate;
                if ($bookingDate >= $offerStartDate && $bookingDate <= $offerEndDate) {
                    // Apply discount
                    $discountAmount = $baseRate * ($discountPercentage / 100);
                    $offerPrice = $baseRate - $discountAmount;
                    $baseRate = round($offerPrice, 2);
                }


                if ($bookingDate >= $surchargeStartDate && $bookingDate <= $surchargeEndDate) {
                    // Apply discount
                    $discountAmount = $baseRate * ($surchargePercentage / 100);
                    $surchargePrice = $baseRate + $discountAmount;
                    $baseRate = round($surchargePrice, 2);
                }
                // if(auth()->user()->role=='Partner'){
                //     $baseRate = $fare->base_rate*(auth()->user()->platform_fee/100);
                // }else{

                //  }

                $additionalKmRate = $fare->additional_km_rate;

                $totalFare = $baseRate * $totalDays;

                if ($distance > $baseKm) {
                    $extraKm = $distance - $baseKm;
                    $totalFare += $extraKm * $additionalKmRate;
                }

                $tax = ($setting->tax / 100) * $totalFare;
                //  $item->basefare = round($baseRate * $totalDays, 2);
                $item->fare = round($totalFare, 2);
                $item->tax_amount = round($tax, 2);
                $item->total_rate = $item->fare + $item->tax_amount;
                $item->platform_fee = ($setting->platform_fee / 100) * $item->total_rate;
            }
            return response()->json([
                'status' => true,
                'total_distance_in_km' => round($distance, 2),
                'serachhistory_id' => $serachhistory->id,
                'cabs' => $vehicleType,

            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function chooseCar(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'origin' => 'required',
                'vehicle_type' => 'required',
            ]);

            $origin = $request->origin;


            // Step 1: Get coordinates of origin
            $originCoords = $this->getCoordinatesFromAddress($origin);

            // Step 2: Find the zone for origin


            $zones = Zone::all();
            $originZone = null;

            foreach ($zones as $zone) {
                $polygon = json_decode($zone->coordinates, true); // ← fix here
                if ($this->isPointInPolygon($originCoords, $polygon)) {
                    $originZone = $zone;
                    break;
                }
            }


            if (!$originZone) {
                return response()->json(['error' => 'Origin is outside service zones'], 422);
            }


            // 3 images
            $images = [];

            $vehicles = Vehicle::join('vendors', 'vehicles.vendor_id', '=', 'vendors.id')
                //  ->join('drivers', 'vehicles.driver_id', '=', 'drivers.id')
                ->leftJoin('reviews', 'vehicles.id', '=', 'reviews.vehicle_id')
                ->leftJoin('booking_histories', 'vehicles.id', '=', 'booking_histories.vehicle_id')
                ->select(
                    'vehicles.id',
                    'vehicles.vehicle_type',
                    'vehicles.seats',
                    'vehicles.fuel_type',
                    'vehicles.model',
                    'vehicles.vehicle_make',
                    'vehicles.image',
                    'vendors.name as vendor_name',
                    'vendors.vendor_image',
                    DB::raw('AVG(reviews.overall_rating) as avg_rating'),
                    DB::raw('COUNT(booking_histories.id) as total_trips')
                )
                ->where('vehicles.vehicle_type', $request->vehicle_type)
                ->where('vehicles.reg_validity', '>=', date("Y-m-d", strtotime(now())))
                ->where('vehicles.insurance_validity', '>=', date("Y-m-d", strtotime(now())))
                ->where('vehicles.tax_validity', '>=', date("Y-m-d", strtotime(now())))
                ->where('vendors.zone', '=', $originZone->id)
                ->where('vendors.status', 'ACTIVE')
                ->groupBy(
                    'vehicles.id',
                    'vehicles.vehicle_type',
                    'vehicles.seats',
                    'vehicles.fuel_type',
                    'vehicles.model',
                    'vehicles.vehicle_make',
                    'vehicles.image',
                    'vendors.name',
                    'vendors.vendor_image'
                );
            // ->having('avg_rating', '>=', $request->filter_rate) // Optional: filter by minimum rating
            //  ->get();
            //  print_r($vehicles);die();
            if ($request->filled('filter_rate')) {
                $vehicles = $vehicles->where('overall_rating', $request->filter_rate);
            }
            if ($request->filled('filter_cab_model')) {
                $vehicles = $vehicles->where('vehicle_make', $request->filter_cab_model);
            }

            $vehicles = $vehicles->get();
            $gallery_images = [];
            if (count($vehicles) > 0) {
                foreach ($vehicles as $vehicle) {
                    if ($vehicle->image != NULL) {
                        $vehicle->model = BrandVehicle::where('id', $vehicle->model)->pluck('model_name')->first();
                        foreach (json_decode($vehicle->image) as $i => $img) {
                            if ($i == 3) {
                                break; // Stop looping once 3 images collected
                            }
                            $gallery_images[$i] = asset('uploads/vehicle/' . $img); // or Storage::url($img) if using storage



                        }
                    }
                    if ($vehicle->vendor_image == NULL) {
                        $vehicle->vendor_image = asset('assets/images/user.jpg');
                    } else {
                        $vehicle->vendor_image = asset('uploads/vendor/' . $vehicle->vendor_image);
                    }
                    $vehicle->image = $gallery_images;
                }
            }


            return response()->json([
                'status' => true,
                'cars' => $vehicles,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }


    public function addCommission(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'commission' => 'required',
                'total_fare' => 'required',
                'tax' => 'required',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Validation errors',
                    'errors' => $validator->errors()
                ], 422);
            }


            $setting = Settings::first();
            $commission_rate =  ($request->commission / 100) * $request->total_fare;
            $total_fare = $request->total_fare + $commission_rate + $request->tax;
            $platform_fee = ($setting->platform_fee / 100) * $total_fare;
            return response()->json([
                'status' => true,
                'commission_rate' => round($commission_rate, 2),
                'subtotal' => round($total_fare, 2),
                'platform_fee' => round($platform_fee, 2)
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function bookCab02(Request $request)
    {
        try {
            $bookingHistory = SearchHistory::where('id', $request->serachhistory_id)->first();


            $latest = BookingHistory::orderBy('id', 'DESC')->first();
            $number = $latest ? intval(substr($latest->booking_id, 2)) + 1 : 1;
            $bookingid = 'BK' . str_pad($number, 6, '0', STR_PAD_LEFT);

            $vehicle = Vehicle::where('id', $request->vehicle_id)->first();

            $res = BookingHistory::insert([
                'booking_id' => $bookingid,
                'vendor_id' => $vehicle->vendor_id,
                'vehicle_id' => $request->vehicle_id,
                //'driver_id'=> $vehicle->driver_id ,
                'booked_user_id' => auth()->user()->id,
                'role' => auth()->user()->role,
                'total_km' => $request->total_km,
                'departure_date' => $bookingHistory->departure_date,
                'return_date' => $bookingHistory->return_date,
                'departure_time' => $bookingHistory->departure_time,
                'return_time' => $bookingHistory->return_time,
                'pick_up' => $bookingHistory->pick_up,
                'drop_off' => $bookingHistory->drop_off,
                'stay_date' => $bookingHistory->stay_date,
                'stay_locations' => $bookingHistory->stay_locations,
                'adults' => $bookingHistory->adults,
                'children' => $bookingHistory->children,
                'client_name' => $bookingHistory->client_name,
                'client_mobile_number' => $bookingHistory->client_mobile_number,
                'client_email' => $bookingHistory->client_email,
                'base_fare' => $request->base_fare,
                'commission_rate' => $request->commission_rate,
                'tax' => $request->tax,
                'total_fare' => $request->total_fare,
                'platform_fee' => $request->platform_fee,
                'paid_amount' => $request->paid_amount,
            ]);
            // ********** stay date  *********
            $raw = $bookingHistory->stay_date; // The double-encoded JSON string

            $firstDecode = json_decode($raw, true);              // returns a JSON string
            $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array

            $days = $locationsArray;

            // ********** stay location  *********
            $raw = $bookingHistory->stay_locations; // The double-encoded JSON string
            $firstDecode = json_decode($raw, true);              // returns a JSON string
            $stay_locations = json_decode($firstDecode, true);   // returns the actual PHP array
            // ********** stay date end  *********

            // ********** stay date end *********

            $origin = $bookingHistory->pick_up;
            $destination = $stay_locations[0];
            $apiKey = Settings::pluck('google_map_api_key')->first();
            $additional_km = Settings::pluck('additional_km')->first();
            for ($i = 0; $i < count($days); $i++) {
                $daycount = 'Day ' . ($i + 1);
                // find total return km


                $response = Http::get("https://maps.googleapis.com/maps/api/directions/json", [
                    'origin' => $origin,
                    'destination' => $destination,
                    'key' => $apiKey
                ]);

                $data = $response->json();

                if ($data['status'] !== 'OK') {
                    return response()->json(['error' => 'Unable to calculate distance', 'status' => $data['status']], 500);
                }

                $distanceInMeters = $data['routes'][0]['legs'][0]['distance']['value'];
                $distanceInKm = round($distanceInMeters / 1000, 2) + $additional_km;

                TripLifeCycle::insert(
                    [
                        'booking_id' => BookingHistory::where('booking_id', $bookingid)->pluck('id')->first(),
                        'date' => date('Y-m-d', strtotime($days[$i])),
                        'day' => $daycount,
                        'actual_km' => $distanceInKm
                    ]
                );

                $origin = $stay_locations[$i];
                if (count($days) == ($i + 1)) {
                    $destination = $bookingHistory->drop_off;
                } else {
                    $destination = $stay_locations[$i + 1];
                }
            }
            if ($res == 1) {

                // ********** sent and save notification *********

                $vendor = Vendors::where('id', $vehicle->vendor_id)->pluck('name')->first();
                $vehicleCategoryName = DB::table('vehicles')
                    ->join('vehicle_categories', 'vehicles.vehicle_type', '=', 'vehicle_categories.id')
                    ->where('vehicles.id', $request->vehicle_id)
                    ->value('vehicle_categories.name');
                $curl = curl_init();

                curl_setopt_array($curl, array(
                    CURLOPT_URL => 'https://api.connectpanels.com/whatsapp-api/v1.0/customer/120180/bot/0d094a19d57e4eaa/template',
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_ENCODING => '',
                    CURLOPT_MAXREDIRS => 10,
                    CURLOPT_TIMEOUT => 0,
                    CURLOPT_FOLLOWLOCATION => true,
                    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                    CURLOPT_CUSTOMREQUEST => 'POST',
                    CURLOPT_POSTFIELDS => '{"payload":{"name":"booking_completion","components":[{"type":"body","parameters":[{"type":"text","text":"' . $bookingid . '"},{"type":"text","text":"' . $vendor . '"},{"type":"text","text":"' . $vehicleCategoryName . '"}]}],"language":{"code":"en_US","policy":"deterministic"},"namespace":"66cbcb46_a232_4519_a6e7_b69fd50991bf"},"phoneNumber":"9188360241"}',
                    CURLOPT_HTTPHEADER => array(
                        'Authorization: Basic c4d8e3eb-4e7c-4f9d-a05c-5ff0bf583d29-Ic1uEZu',
                        'Content-Type: application/json'
                    ),
                ));

                $response = curl_exec($curl);

                curl_close($curl);
                // echo $response;
                Notifications::insert([
                    'booking_id' => $bookingid,
                    'vendor_id' => $vehicle->vendor_id,
                    'driver_id' => $vehicle->driver_id,
                    'booked_user_id' => auth()->user()->id,
                    'notification' => "💛 Your Booking is Confirmed!

📌  Booking ID: BK000021
🏢 Vendor: Aminesh
🚙  Vehicle Type: SEDAN"
                ]);

                // ********** end sent and save notification *********

                return response()->json([
                    'status' => true,
                    'message' => 'Ride booked successfully!',
                ]);
            }
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }



    public function getBookings(Request $request)
    {
        try {

            $bookings = BookingHistory::select('id', 'booking_id', 'departure_date', 'return_date', 'departure_time', 'return_time', 'pick_up', 'drop_off', 'stay_locations', 'booking_status', 'total_fare', 'paid_amount')
                ->where('booked_user_id', auth()->user()->id);
            if ($request->filled('status')) {
                $bookings = $bookings->where('booking_status', $request->status);
            }
            $bookings = $bookings->get();
            foreach ($bookings as $item) {
                $raw = $item->stay_locations; // The double-encoded JSON string

                $firstDecode = json_decode($raw, true);              // returns a JSON string
                $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array

                $locationString = is_array($locationsArray) ? implode(', ', $locationsArray) : '';

                $item->stay_locations = $locationString;
                //  $item->stay_locations_count = is_array($locationsArray) ? count($locationsArray) : 0;

                $departure = Carbon::parse($item->departure_date);
                $return = Carbon::parse($item->return_date);

                // Total days (inclusive)
                $totalDays = $departure->diffInDays($return) + 1; // +1 to include both start and end dates
                $nightCount = $totalDays - 1;

                // Final title
                $tripTitle = "{$nightCount}NIGHTS {$totalDays}DAYS";
                $item->title = $tripTitle;
            }

            return response()->json([
                'status' => true,
                'bookings' => $bookings,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getBookingDetails(Request $request)
    {
        try {

            // $booking= BookingHistory::select('id','vehicle_id','driver_id','booking_id','total_km','departure_date','return_date','departure_time','return_time','pick_up','drop_off','stay_date','stay_locations','adults','children','client_name','client_mobile_number','client_email','booking_status','total_fare','paid_amount','overall_review_rate')
            // ->where('id',$request->booking_id)
            // ->first();

            $booking =  BookingHistory::select(
                'booking_histories.*',
                'reviews.driver_rating',
                'reviews.car_rating',
                'reviews.comment'
            )
                ->leftJoin('reviews', 'booking_histories.id', '=', 'reviews.booking_id')
                ->where('booking_histories.id', $request->booking_id)
                ->first();
            $booking->makeHidden(['created_at', 'updated_at']);
            // ********** stay location  *********
            $raw = $booking->stay_locations; // The double-encoded JSON string
            $firstDecode = json_decode($raw, true);              // returns a JSON string
            $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array
            $booking->stay_locations = $locationsArray;
            // ********** stay date end  *********

            // ********** find titile  *********
            $departure = Carbon::parse($booking->departure_date);
            $return = Carbon::parse($booking->return_date);

            // Total days (inclusive)
            $totalDays = $departure->diffInDays($return) + 1; // +1 to include both start and end dates
            $nightCount = $totalDays - 1;

            // Final title
            $tripTitle = "{$nightCount}NIGHTS {$totalDays}DAYS";
            $booking->title = $tripTitle;
            // ********** find titile end  *********

            // ********** stay date  *********
            $raw = $booking->stay_date; // The double-encoded JSON string

            $firstDecode = json_decode($raw, true);              // returns a JSON string
            $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array

            $booking->stay_date = $locationsArray;
            // ********** stay date end *********
            $vehicle = Vehicle::join('drivers', 'vehicles.driver_id', '=', 'drivers.id')
                ->join('vendors', 'vehicles.vendor_id', '=', 'vendors.id')
                ->leftJoin('reviews', 'vehicles.id', '=', 'reviews.vehicle_id')
                ->leftJoin('booking_histories', 'vehicles.id', '=', 'booking_histories.vehicle_id')
                ->select(
                    'vehicles.id',
                    'vehicles.seats',
                    'vehicles.fuel_type',
                    'vehicles.model',
                    'vehicles.vehicle_make',
                    'vehicles.image',
                    'vendors.name as vendor_name',
                    'vendors.vendor_image',
                    'drivers.name as driver_name',
                    'drivers.email as driver_email',
                    'drivers.phone as driver_phone',
                    DB::raw('AVG(reviews.overall_rating) as avg_rating'),
                    DB::raw('COUNT(booking_histories.id) as total_trips')
                )
                ->where('vehicles.id', $booking->vehicle_id)
                ->groupBy(
                    'vehicles.id',
                    'vehicles.seats',
                    'vehicles.fuel_type',
                    'vehicles.model',
                    'vehicles.vehicle_make',
                    'vehicles.image',
                    'vendors.name',
                    'vendors.vendor_image',
                    'drivers.name',
                    'drivers.email',
                    'drivers.phone',
                )
                ->first();
            // $vehicle =  Vehicle::join('vendors', 'vehicles.vendor_id', '=', 'vendors.id')
            //                ->join('drivers', 'drivers.vendor_id', '=', 'vendors.id')
            //                 ->select('vehicles.id','vehicles.seats','vehicles.fuel_type','vehicles.model','vehicles.vehicle_make','vehicles.image', 'vendors.name as vendor_name','vendors.vendor_image') // select vendor fields as needed
            //                 ->where('vehicles.id',$booking->vehicle_id)
            //                 ->first();

            $vehicle->model = BrandVehicle::where('id', $vehicle->model)->pluck('model_name')->first();
            foreach (json_decode($vehicle->image) as $i => $img) {
                if ($i == 3) {
                    break; // Stop looping once 3 images collected
                }
                $gallery_images[$i] = asset('uploads/vehicle/' . $img); // or Storage::url($img) if using storage
            }

            if ($vehicle->vendor_image == NULL) {
                $vehicle->vendor_image = asset('assets/images/user.jpg');
            } else {
                $vehicle->vendor_image = asset('uploads/vendor/' . $vehicle->vendor_image);
            }
            $vehicle->image = $gallery_images;
            $vehicle->review_rate = 5;


            $booking->cab_agency_details = $vehicle;
            return response()->json([
                'status' => true,
                'booking_details' => $booking,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }


    public function addReview(Request $request)
    {
        try {
            $exist = Review::where('booking_id', $request->booking_id)->first();
            if ($exist == null) {
                $res = Review::insert([
                    'booking_id' => $request->booking_id,
                    'user_id' => auth()->user()->id,
                    'vehicle_id' => $request->vehicle_id,
                    'driver_id' => $request->driver_id,
                    'overall_rating' => $request->overall_rating,
                    'driver_rating' => $request->driver_rating,
                    'car_rating' => $request->car_rating,
                    'comment' => $request->comment,
                ]);
            } else {
                $res = Review::where('id', $exist->id)->update([
                    'booking_id' => $request->booking_id,
                    'user_id' => auth()->user()->id,
                    'vehicle_id' => $request->vehicle_id,
                    'driver_id' => $request->driver_id,
                    'overall_rating' => $request->overall_rating,
                    'driver_rating' => $request->driver_rating,
                    'car_rating' => $request->car_rating,
                    'comment' => $request->comment,
                ]);
            }
            BookingHistory::where('id', $request->booking_id)->update(['overall_review_rate' => $request->overall_rating]);
            if ($res == 1) {
                return response()->json([
                    'status' => true,
                    'message' => "Review added Successfully",
                ]);
            }
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getAgencyDetails(Request $request)
    {
        try {
            $vehicle =  Vehicle::join('vendors', 'vehicles.vendor_id', '=', 'vendors.id')
                ->join('drivers', 'drivers.vendor_id', '=', 'vendors.id')
                ->select(
                    'vehicles.id',
                    'vehicles.model',
                    'vehicles.vehicle_make',
                    'vehicles.year_of_registration',
                    'vehicles.seats',
                    'vehicles.color',
                    'vehicles.fuel_type',
                    'vehicles.plate_number',
                    'vehicles.reg_validity',
                    'vehicles.insurance_validity',
                    'vehicles.tax_validity',
                    'vehicles.image',
                    'vehicles.vehicle_type',
                    'vendors.name as vendor_name',
                    'vendors.vendor_image'
                ) // select vendor fields as needed
                ->where('vehicles.id', $request->vehicle_id)
                ->first();
            $vehicle->model = BrandVehicle::where('id', $vehicle->model)->pluck('model_name')->first();
            foreach (json_decode($vehicle->image) as $i => $img) {
                if ($i == 3) {
                    break; // Stop looping once 3 images collected
                }
                $gallery_images[$i] = asset('uploads/vehicle/' . $img); // or Storage::url($img) if using storage
            }

            if ($vehicle->vendor_image == NULL) {
                $vehicle->vendor_image = asset('assets/images/user.jpg');
            } else {
                $vehicle->vendor_image = asset('uploads/vendor/' . $vehicle->vendor_image);
            }
            $vehicle->image = $gallery_images;
            //  $vehicle->reviews=Review::where('vehicle_id',$request->vehicle_id)->get(); 
            $vehicle->reviews =  Review::select('reviews.id as reviewid', 'partners.name as user_name', 'partners.image as user_image', 'reviews.overall_rating', 'reviews.driver_rating', 'reviews.car_rating', 'reviews.comment', 'reviews.created_at')
                ->join('partners', 'partners.id', '=', 'reviews.user_id')
                ->where('reviews.vehicle_id', $request->vehicle_id)
                ->orderBy('reviews.id', 'desc')
                ->get();
            foreach ($vehicle->reviews as $item) {

                if ($item->image == NULL) {
                    $item->image = asset('assets/images/user.jpg');
                } else {
                    $item->image = asset('uploads/partner/' . $item->image);
                }
            }
            $vehicle->vehicle_type = VehicleCategory::where('id', $vehicle->vehicle_type)->pluck('name')->first();

            //** rating **** */

            $perStarAverage = Review::select(
                'overall_rating',
                DB::raw('ROUND(AVG(overall_rating), 1) as average'),
                DB::raw('COUNT(*) as count')
            )
                ->whereIn('overall_rating', [1, 2, 3, 4, 5])
                ->where('vehicle_id', $request->vehicle_id)
                ->groupBy('overall_rating')
                ->orderBy('overall_rating', 'desc')
                ->get();

            $vehicle->total_review_rate = $perStarAverage;

            // Initialize all ratings from 0 to 5 with default values
            $ratings = [];
            for ($i = 0; $i <= 5; $i++) {
                $ratings[$i] = [
                    'rate' => $i,
                    'average' => 0.0,
                    'count' => 0,
                ];
            }

            // Overwrite with actual data from DB
            foreach ($perStarAverage as $row) {
                $ratings[$row->overall_rating] = [
                    'rate' => $row->overall_rating,
                    'average' => (float) $row->average,
                    'count' => (int) $row->count,
                ];
            }

            //  $perStarAverage->review_details=$ratings;
            $vehicle->review_star_rate = $ratings;
            return response()->json([
                'status' => true,
                'vehicle' => $vehicle,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getVehicleModels(Request $request)
    {
        try {

            $models = Vehicle::distinct()->where('status', 'ACTIVE')->pluck('vehicle_make');


            return response()->json([
                'status' => true,
                'models' => $models,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getRecentSearch(Request $request)
    {
        try {

            $bookings = SearchHistory::select('id', 'departure_date', 'return_date', 'departure_time', 'return_time', 'pick_up', 'drop_off')
                ->where('user_id', auth()->user()->id)
                ->orderBy('id', 'DESC')
                ->get();

            return response()->json([
                'status' => true,
                'data' => $bookings,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getRecentSearchDetails(Request $request)
    {
        try {

            $booking = SearchHistory::where('id', $request->search_id)
                ->first();

            // ********** stay location  *********
            $raw = $booking->stay_locations; // The double-encoded JSON string
            $firstDecode = json_decode($raw, true);              // returns a JSON string
            $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array
            $booking->stay_locations = $locationsArray;
            // ********** stay date end  *********

            // ********** find titile  *********
            $departure = Carbon::parse($booking->departure_date);
            $return = Carbon::parse($booking->return_date);

            // Total days (inclusive)
            $totalDays = $departure->diffInDays($return) + 1; // +1 to include both start and end dates
            $nightCount = $totalDays - 1;

            // Final title
            $tripTitle = "{$nightCount}NIGHTS {$totalDays}DAYS";
            $booking->title = $tripTitle;
            // ********** find titile end  *********

            // ********** stay date  *********
            $raw = $booking->stay_date; // The double-encoded JSON string

            $firstDecode = json_decode($raw, true);              // returns a JSON string
            $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array

            $booking->stay_date = $locationsArray;
            // ********** stay date end *********
            return response()->json([
                'status' => true,
                'data' => $booking,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function banners(Request $request)
    {
        try {
            $banners = Banners::select('id', 'link', 'photo')->get();

            foreach ($banners  as $k => $item) {
                $item->photo = asset("uploads/banner/" . $item->photo);
            }
            return response()->json([
                'status' => true,
                'data' => $banners,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function downloadInvoice(Request $request)
    {

        $data = BookingHistory::join('vehicles', 'vehicles.id', '=', 'booking_histories.vehicle_id')
            ->join('brand_vehicles', 'vehicles.model', '=', 'brand_vehicles.id')
            ->leftJoin('drivers', 'booking_histories.driver_id', '=', 'drivers.id') // changed to leftJoin
            ->select(
                'booking_histories.*',
                'drivers.name',
                'brand_vehicles.model_name',
                'vehicles.plate_number',
            )
            ->where('booking_histories.booking_id', $request->id)
            ->first();

        // ********** stay location  *********
        $raw = $data->stay_locations; // The double-encoded JSON string
        $firstDecode = json_decode($raw, true);              // returns a JSON string
        $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array
        $data->stay_locations = $locationsArray;
        // ********** stay date end  *********

        // Sample Data (replace with DB call)
        $invoice = (object)[
            'booking_id' => $data->booking_id,
            'trip_date' =>  $data->departure_date,
            'driver_name' =>  $data->name,
            'vehicle' =>  $data->model_name,
            'vehicle_no' => $data->plate_number,
            'customer_name' => $data->client_name,
            'customer_contact' => $data->client_mobile_number,
            'pickup_address' => $data->pick_up,
            'drop_address' => $data->drop_off,
            'base_fare' => $data->base_fare,
            'tax_percentage' => Settings::pluck('tax')->first(),
            'tax' => $data->tax,
            'total_fare' => $data->total_fare,
            'platform_fee' => $data->platform_fee,
            'paid_amount' => $data->paid_amount,
            'pickup_time' => $data->departure_time,
            'drop_time' => $data->return_time,
            'total_km' => $data->total_km,
            'days' => count($data->stay_locations),
            'payment_mode' => 'UPI',
            'status' => $data->booking_status
        ];

        //  return view('invoice', compact('invoice'));
        $pdf = Pdf::loadView('invoice', compact('invoice'));

        return response($pdf->output(), 200)
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="invoice_' . $invoice->booking_id . '.pdf"');

        //    return $pdf->download('invoice_' . $invoice->booking_id . '.pdf');
    }

    public function createRazorpayOrder(Request $request)
    {
        $request->validate([
            'paid_amount' => 'required|numeric',
            //  'booking_id' => 'required|string'
        ]);
        $RAZORPAY_KEY = Settings::pluck('razor_key_id')->first();
        $RAZORPAY_SECRET = Settings::pluck('razor_key_secret')->first();
        $api = new Api($RAZORPAY_KEY, $RAZORPAY_SECRET);

        $order = $api->order->create([
            'receipt' => 'RCPT-' . now()->format('Ymd-His'),
            'amount' => $request->paid_amount * 100,
            'currency' => 'INR'
        ]);

        return response()->json([
            'razorpay_order_id' => $order->id,
            //   'razorpay_key' => $RAZORPAY_KEY,
            'amount' => $order->amount,
        ]);
    }

    public function bookCab(Request $request)
    {
        try {
            $RAZORPAY_KEY = Settings::pluck('razor_key_id')->first();
            $RAZORPAY_SECRET = Settings::pluck('razor_key_secret')->first();

            $request->validate([
                'razorpay_payment_id' => 'required|string',
                'razorpay_order_id' => 'required|string',
                'razorpay_signature' => 'required|string',
                'vehicle_id' => 'required|integer',
                'serachhistory_id' => 'required|integer',
                'total_km' => 'required|numeric',
                'base_fare' => 'required|numeric',
                'commission_rate' => 'required|numeric',
                'tax' => 'required|numeric',
                'total_fare' => 'required|numeric',
                'platform_fee' => 'required|numeric',
                'paid_amount' => 'required|numeric',
            ]);

            // 1️⃣ Verify Razorpay Signature
            $generatedSignature = hash_hmac(
                'sha256',
                $request->razorpay_order_id . "|" . $request->razorpay_payment_id,
                $RAZORPAY_SECRET
            );

            if ($generatedSignature !== $request->razorpay_signature) {
                return response()->json(['error' => 'Payment verification failed'], 403);
            }

            $api = new Api($RAZORPAY_KEY, $RAZORPAY_SECRET);


            $payment = $api->payment->fetch($request->razorpay_payment_id);

            $razorpayPaymentStatus = $payment['status'] ?? 'unknown';

            // 2️⃣ Get data from SearchHistory
            $bookingHistory = SearchHistory::find($request->serachhistory_id);
            $latest = BookingHistory::latest('id')->first();
            $number = $latest ? intval(substr($latest->booking_id, 2)) + 1 : 1;
            $bookingid = 'BK' . str_pad($number, 6, '0', STR_PAD_LEFT);
            $vehicle = Vehicle::find($request->vehicle_id);

            // 3️⃣ Insert into booking_histories
            $bookingRes = BookingHistory::create([
                'booking_id' => $bookingid,
                'vendor_id' => $vehicle->vendor_id,
                'vehicle_id' => $request->vehicle_id,
                'booked_user_id' => auth()->user()->id,
                'role' => auth()->user()->role,
                'total_km' => $request->total_km,
                'departure_date' => $bookingHistory->departure_date,
                'return_date' => $bookingHistory->return_date,
                'departure_time' => $bookingHistory->departure_time,
                'return_time' => $bookingHistory->return_time,
                'pick_up' => $bookingHistory->pick_up,
                'drop_off' => $bookingHistory->drop_off,
                'stay_date' => $bookingHistory->stay_date,
                'stay_locations' => $bookingHistory->stay_locations,
                'adults' => $bookingHistory->adults,
                'children' => $bookingHistory->children,
                'client_name' => $bookingHistory->client_name,
                'client_mobile_number' => $bookingHistory->client_mobile_number,
                'client_email' => $bookingHistory->client_email,
                'base_fare' => $request->base_fare,
                'commission_rate' => $request->commission_rate,
                'tax' => $request->tax,
                'total_fare' => $request->total_fare,
                'platform_fee' => $request->platform_fee,
                'paid_amount' => $request->paid_amount,
                'payment_id' => $request->razorpay_payment_id,
                'payment_status' => 'Paid'
            ]);

            // 4️⃣ Insert into TripLifeCycle (same as your original code)
            $days = json_decode(json_decode($bookingHistory->stay_date, true), true);
            $stay_locations = json_decode(json_decode($bookingHistory->stay_locations, true), true);
            $origin = $bookingHistory->pick_up;
            $destination = $stay_locations[0];
            $apiKey = Settings::pluck('google_map_api_key')->first();
            $additional_km = Settings::pluck('additional_km')->first();

            for ($i = 0; $i < count($days); $i++) {
                $daycount = 'Day ' . ($i + 1);

                $response = Http::get("https://maps.googleapis.com/maps/api/directions/json", [
                    'origin' => $origin,
                    'destination' => $destination,
                    'key' => $apiKey
                ]);

                $data = $response->json();

                if ($data['status'] !== 'OK') {
                    return response()->json(['error' => 'Unable to calculate distance', 'status' => $data['status']], 500);
                }

                $distanceInKm = round($data['routes'][0]['legs'][0]['distance']['value'] / 1000, 2) + $additional_km;

                TripLifeCycle::create([
                    'booking_id' => $bookingRes->id,
                    'date' => date('Y-m-d', strtotime($days[$i])),
                    'day' => $daycount,
                    'actual_km' => $distanceInKm
                ]);

                $origin = $stay_locations[$i];
                $destination = $i + 1 == count($stay_locations) ? $bookingHistory->drop_off : $stay_locations[$i + 1];
            }

           // ********** sent and save notification *********

                $vendor = Vendors::where('id', $vehicle->vendor_id)->pluck('name')->first();
                $vehicleCategoryName = DB::table('vehicles')
                    ->join('vehicle_categories', 'vehicles.vehicle_type', '=', 'vehicle_categories.id')
                    ->where('vehicles.id', $request->vehicle_id)
                    ->value('vehicle_categories.name');
                $curl = curl_init();

                curl_setopt_array($curl, array(
                    CURLOPT_URL => 'https://api.connectpanels.com/whatsapp-api/v1.0/customer/120180/bot/0d094a19d57e4eaa/template',
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_ENCODING => '',
                    CURLOPT_MAXREDIRS => 10,
                    CURLOPT_TIMEOUT => 0,
                    CURLOPT_FOLLOWLOCATION => true,
                    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                    CURLOPT_CUSTOMREQUEST => 'POST',
                    CURLOPT_POSTFIELDS => '{"payload":{"name":"booking_completion","components":[{"type":"body","parameters":[{"type":"text","text":"' . $bookingid . '"},{"type":"text","text":"' . $vendor . '"},{"type":"text","text":"' . $vehicleCategoryName . '"}]}],"language":{"code":"en_US","policy":"deterministic"},"namespace":"66cbcb46_a232_4519_a6e7_b69fd50991bf"},"phoneNumber":"'.$bookingHistory->client_mobile_number.'"}',
                    CURLOPT_HTTPHEADER => array(
                        'Authorization: Basic c4d8e3eb-4e7c-4f9d-a05c-5ff0bf583d29-Ic1uEZu',
                        'Content-Type: application/json'
                    ),
                ));

                $response = curl_exec($curl);

                curl_close($curl);
                // echo $response;
                Notifications::insert([
                    'booking_id' => $bookingid,
                    'vendor_id' => $vehicle->vendor_id,
                    'driver_id' => $vehicle->driver_id,
                    'booked_user_id' => auth()->user()->id,
                    'notification' => "💛 Your Booking is Confirmed!

                                    📌  Booking ID: BK000021
                                    🏢 Vendor: Aminesh
                                    🚙  Vehicle Type: SEDAN"
                ]);

                // ********** end sent and save notification *********


            return response()->json([
                'status' => true,
                'message' => 'Booking confirmed and payment verified successfully!',
                'booking_id' => $bookingid
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e->getMessage()], 500);
        }
    }

    public function notifications(Request $request)
    {
        try {
            $data = Notifications::where('booked_user_id', auth()->user()->id)
                ->get();

            return response()->json([
                'status' => true,
                'data' => $data,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }
}
